import uuid
from datetime import datetime, timedelta

import ezcord


class SessionDB(ezcord.DBHandler):
    def __init__(self):
        super().__init__("essentials/dashboard.db")

    async def setup(self):
        await self.exec(
            """CREATE TABLE IF NOT EXISTS sessions (
            session_id TEXT UNIQUE,
            token TEXT,
            refresh_token TEXT,
            token_expires_at TIMESTAMP,
            user_id INTEGER PRIMARY KEY
            )"""
        )

    async def add_session(self, token, refresh_token, expires_in, user_id):
        session_id = str(uuid.uuid4())
        expires = datetime.now() + timedelta(seconds=expires_in)

        await self.exec(
            """INSERT OR REPLACE INTO sessions (session_id, token, refresh_token, token_expires_at, user_id)
            VALUES (?, ?, ?, ?, ?)""",
            (session_id, token, refresh_token, expires, user_id),
        )
        return session_id

    async def get_session(self, session_id):
        return await self.one(
            "SELECT token, refresh_token, token_expires_at, user_id FROM sessions WHERE session_id = ?",
            session_id,
            detect_types=1
        )
    
    async def update_session(self, session_id, token, refresh_token, token_expires_at):
        await self.exec(
            "UPDATE sessions SET token = ?, refresh_token = ?, token_expires_at = ? WHERE session_id = ?",
            (token, refresh_token, token_expires_at, session_id),
            detect_types=1
        )
    
    async def delete_session(self, session_id):
        await self.exec("DELETE FROM sessions WHERE session_id = ?", session_id)

db = SessionDB()



class UserDB(ezcord.DBHandler):
    def __init__(self):
        super().__init__("essentials/dashboard.db")

    async def setup(self):
        await self.exec(
            """CREATE TABLE IF NOT EXISTS users (
            user_id INTEGER PRIMARY KEY,
            rank TEXT
            )"""
        )

    async def add_user(self, user_id, rank):
        await self.exec(
            """INSERT OR REPLACE INTO users (user_id, rank)
            VALUES (?, ?)""",
            (user_id, rank)
        )

    async def get_user(self, user_id, rank):
        await self.one(
            "SELECT * FROM users",
            detect_types=1
        )

    async def update_user(self, user_id, rank):
        await self.exec(
            "UPDATE users SET rank = ? WHERE user_id = ?",
            (rank, user_id),
            detect_types=1
        )

    async def delete_user(self, user_id):
        await self.exec("DELETE FROM users WHERE user_id = ?", user_id)

user = UserDB()



class ServerDB(ezcord.DBHandler):
    def __init__(self):
        super().__init__("essentials/dashboard.db")

    async def setup(self):
        await self.exec(
            """CREATE TABLE IF NOT EXISTS server (
            server_id INTEGER PRIMARY KEY,
            language TEXT,
            blacklisted BOOLEAN
            )"""
        )

    async def add_server(self, server_id):
        await self.exec(
            "INSERT OR IGNORE INTO server (server_id, language, blacklisted) VALUES (?, ?, ?)",
            (server_id, "None", False)
        )

server = ServerDB()



class BotPluginsDB(ezcord.DBHandler):
    def __init__(self):
        super().__init__("essentials/dashboard.db")

    async def setup(self):
        await self.exec(
            """CREATE TABLE IF NOT EXISTS bot_plugins (
            plugin_id INTEGER PRIMARY KEY AUTOINCREMENT,
            plugin_name TEXT UNIQUE,
            state TEXT
            )"""
        )

    async def get_plugins(self, plugin_name):
        return await self.one(
            "SELECT * FROM bot_plugins WHERE plugin_name = ?",
            (plugin_name)
        )

    async def add_plugin(self, plugin):
        await self.exec(
            "INSERT OR IGNORE INTO bot_plugins (plugin_name, state) VALUES (?, ?)",
            (str(plugin), "maintenance")
        )

    async def update_state(self, plugin, state):
        await self.exec(
            "UPDATE bot_plugins SET state = ? WHERE plugin_name = ?",
            (state, plugin),
            detect_types=1
        )

    async def delete_plugin(self, plugin):
        await self.exec(
            "DELETE FROM bot_plugins WHERE plugin_name = ?",
            (plugin)
        )

bot_plugins = BotPluginsDB()



#class PluginsDB(ezcord.DBHandler):
#    def __init__(self):
#        super().__init__("essentials/dashboard.db")
#
#    async def setup(self):
#        await self.exec(
#            """CREATE TABLE IF NOT EXISTS plugins (
#            server_id INTEGER PRIMARY KEY,
#            plugin_id TEXT,
#            state TEXT
#            )"""
#        )
#
#plugin = PluginsDB()




#class HelpDB(ezcord.DBHandler):
#    def __init__(self):
#        super().__init__("essentials/dashboard.db")
#
#    async def setup(self):
#        await self.exec(
#            """CREATE TABLE IF NOT EXISTS help (
#            session_id TEXT UNIQUE,
#            token TEXT,
#            refresh_token TEXT,
#            token_expires_at TIMESTAMP,
#            user_id INTEGER PRIMARY KEY
#            )"""
#        )
#
#help = HelpDB()