from datetime import datetime

import discord
import uvicorn
import os

from discord.ext.ipc import Client
from fastapi import FastAPI, HTTPException, Request, Cookie
from fastapi.responses import RedirectResponse
from fastapi.staticfiles import StaticFiles
from starlette.templating import Jinja2Templates
from dotenv import load_dotenv

from essentials import DiscordAuth, db



app = FastAPI()

app.mount("/scripts", StaticFiles(directory="scripts"), name="scripts")

app.mount("/static", StaticFiles(directory="static"), name="static")

templates = Jinja2Templates(directory="pages")

load_dotenv()
ipc = Client(secret_key="SECRET_KEY")
api = DiscordAuth(os.getenv("CLIENT_ID"), os.getenv("CLIENT_SECRET"), os.getenv("REDIRECT_URI"), os.getenv("API_ENDPOINT"))


@app.on_event("startup")
async def on_startup():
    await api.setup()
    await db.setup()



@app.get("/")
async def home(request: Request):
    return templates.TemplateResponse(
        "index.html",
                {
            "request": request,
            "login_url":os.getenv("LOGIN_URL")
        }
    )



@app.get("/callback")
async def callback(code: str):
    data = {
        "client_id": os.getenv("CLIENT_ID"),
        "client_secret": os.getenv("CLIENT_SECRET"),
        "grant_type": "authorization_code",
        "code": code,
        "redirect_uri": os.getenv("REDIRECT_URI"),
    }

    result = await api.get_token(data)

    if result is None:
        raise HTTPException(status_code=401, detail="Invalid Auth Code")
    
    token, refresh_token, expires_in = result
    user = await api.get_user(token)
    user_id = user.get("id")

    session_id = await db.add_session(token, refresh_token, expires_in, user_id)

    response = RedirectResponse(url="/notpublished")
    response.set_cookie(key="session_id", value=session_id, httponly=True)

    #response = RedirectResponse(url="/guilds")
    return response



@app.get("/notpublished")
async def notpublished(request: Request):
    session_id = request.cookies.get("session_id")
    session = await db.get_session(session_id)
    if not session_id or not session:
        raise HTTPException(status_code=401, detail="no auth")
    
    token, refresh_token, token_expires_at, user_id = session

    user = await api.get_user(token)

    if datetime.now() > token_expires_at or user.get("code") == 0:
        if not await api.refresh_token(session_id, refresh_token):
            RedirectResponse(url="/logout")

    if "id" not in user:
        return RedirectResponse(url="/logout")
    

    return templates.TemplateResponse(
        "notpublished.html",
                {
            "request": request,
            "global_name": user["global_name"]
        }
    )



@app.get("/maintenance")
async def maintenance(request: Request):
    session_id = request.cookies.get("session_id")
    session = await db.get_session(session_id)
    if not session_id or not session:
        raise HTTPException(status_code=401, detail="no auth")
    
    token, refresh_token, token_expires_at, user_id = session

    user = await api.get_user(token)

    if datetime.now() > token_expires_at or user.get("code") == 0:
        if not await api.refresh_token(session_id, refresh_token):
            RedirectResponse(url="/logout")

    if "id" not in user:
        return RedirectResponse(url="/logout")
    

    return templates.TemplateResponse(
        "maintenance.html",
                {
            "request": request,
            "global_name": user["global_name"]
        }
    )



@app.get("/guilds")
async def guilds(request: Request):
    session_id = request.cookies.get("session_id")
    session = await db.get_session(session_id)
    if not session_id or not session:
        raise HTTPException(status_code=401, detail="no auth")
    
    token, refresh_token, token_expires_at, user_id = session

    user = await api.get_user(token)

    if datetime.now() > token_expires_at or user.get("code") == 0:
        if await api.refresh_token(session_id, refresh_token):
            RedirectResponse(url="/guilds")
        else:
            RedirectResponse(url="/logout")

    if "id" not in user:
        return RedirectResponse(url="/logout")

    user_guilds = await api.get_guilds(token)

    permissions = []

    for guild in user_guilds:
        guild["url"] = "/dashboard/" + str(guild["id"])

        if guild["icon"]:
            guild["icon"] = "https://cdn.discordapp.com/icons/" + guild["id"] + "/" + guild["icon"]
        else:
            guild["icon"] = "https://cdn.discordapp.com/embed/avatars/0.png"

        if guild["owner"]:
            guild["level"] = "Owner"
            permissions.append(guild)

        elif discord.Permissions(int(guild["permissions"])).administrator:
            guild["level"] = "Admin"
            permissions.append(guild)


    return templates.TemplateResponse(
        "guilds.html",
                {
            "request": request,
            "global_name": user["global_name"],
            "guilds": permissions
        }
    )



@app.get("/dashboard/{guild_id}")
async def dashboard(request: Request, guild_id: int):
    session_id = request.cookies.get("session_id")
    if not session_id or not await db.get_session(session_id):
        raise HTTPException(status_code=401, detail="no auth")
    
    return templates.TemplateResponse(
        "dashboard.html",
                {
            "request": request,
            "id": guild_id
        }
    )



@app.get("/dashboard/admin/{guild_id}")
async def admindashboard(request: Request, guild_id: int):
    session_id = request.cookies.get("session_id")
    if not session_id or not await db.get_session(session_id):
        raise HTTPException(status_code=401, detail="no auth")
    
    return templates.TemplateResponse(
        "admin_dashboard.html",
                {
            "request": request,
            "id": guild_id
        }
    )



@app.get("/logout")
async def logout(session_id: str = Cookie(None)):
    session = await db.get_session(session_id)
    if not session_id or not session:
        raise HTTPException(status_code=401, detail="no auth")

    token, _, _, _ = session

    response = RedirectResponse("/")
    response.delete_cookie(key="session_id", httponly=True)

    await db.delete_session(session_id)
    await api.revoke_token(token)

    return response



if __name__ == "__main__":
    load_dotenv()
    uvicorn.run("main:app", host="0.0.0.0", port=8500, reload=True)